/*************************************************************************************************************
 *                                                                                                           *
 *  Copyright (c) 2015, Intel Corporation                                                                    *
 *                                                                                                           *
 *  Redistribution and use in source and binary forms, with or without                                       *
 *  modification, are permitted provided that the following conditions are met:                              *
 *                                                                                                           *
 *      * Redistributions of source code must retain the above copyright notice,                             *
 *        this list of conditions and the following disclaimer.                                              *
 *      * Redistributions in binary form must reproduce the above copyright                                  *
 *        notice, this list of conditions and the following disclaimer in the                                *
 *        documentation and/or other materials provided with the distribution.                               *
 *      * Neither the name of Intel Corporation nor the names of its contributors                            *
 *        may be used to endorse or promote products derived from this software                              *
 *        without specific prior written permission.                                                         *
 *                                                                                                           *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"                              *
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE                                *
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE                           *
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE                              *
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL                               *
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR                               *
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER                               *
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,                            *
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE                            *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                     *
 *                                                                                                           *
 *************************************************************************************************************
 *                                                                                                           *
 *  Module name:                                                                                             *
 *      freebsdnaldriver.c                                                                                   *
 *                                                                                                           *
 *  Abstract:                                                                                                *
 *      This file contains sources for functions that handles the NAL driver                                 *
 *                                                                                                           *
 ************************************************************************************************************/

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <freebsdnalioctl.h>
#include <freebsdnaldriver.h>
#include <freebsddevice_i.h>
#include <freebsdos_i.h>
#include <freebsddefs.h>
#include <freebsdtypes.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <nalioctl.h>

/***************************************************************************
 * Local variables definitions
 ***************************************************************************/
struct mtx                          Global_AtomicTestSetSpinLock;
BOOLEAN                             Global_EnabledDebugPrint    = FALSE;
NAL_OS_SPEC_ADAPTER_IN_USE_TABLE    Global_AdapterInUse[NAL_OS_SPEC_MAX_PCI_DEVICES];
UINT32                              Global_DriverReferenceCount = 0;
struct cdev *                       Global_NalDevicePointer     = NULL;
struct cdevsw                       Global_DeviceEntryPoints    = {
        .d_version  = D_VERSION ,           /* version of kernel driver mode */
        .d_name     = "nal" ,               /* name of the driver in /dev/ directory */
        .d_open     = NalOpenDriver ,       /* handler for opening driver */
        .d_close    = NalReleaseDriver ,    /* handler for closing driver */
        .d_ioctl    = NalHandleIoctl        /* handler for ioctl calls */
};

/* Definition of module driver */
DEV_MODULE( nal , NalHandleModuleEvent , NULL );

/***************************************************************************
**
** Name:            NalHandleModuleEvent()
**
** Description:     This is the function, that handles driver module events.
**                  It loads and unloads the nal driver module.
**
** Arguments:       Module  - Pointer to module structure
**                  Event   - Event id to handle
**                  Argument-
**
** Returns:         Code of error from errno.h
**                      0           - when success
**                      EDOOFUS     - when there is error in function implementation
**                      EOPNOTSUPP  - when an event is not recognized
**                      ENXIO       - when device was not loaded before
**
****************************************************************************/
int
NalHandleModuleEvent (
                      IN      module_t    Module ,
                      IN      int         Event ,
                      IN OUT  void *      Argument )
{
  int ErrorCode = EDOOFUS;  /* EDOOFUS - programming error */

  switch(Event)
  {
  /***************************************************************************
   *
   * This case is executed when an event is not recognized
   *
   ***************************************************************************/
  default:
       ErrorCode = EOPNOTSUPP;  /* EOPNOTSUPP - operation not supported */
       break;
  /***************************************************************************
    *
    * This case is executed when the system is about to shut down.
    *
   ***************************************************************************/
  case MOD_SHUTDOWN:
      ErrorCode = EOPNOTSUPP;  /* EOPNOTSUPP - operation not supported */
      break;
  /***************************************************************************
   *
   * This case is executed when the NAL driver should be loaded to the system.
   * It initializes module variables to work.
   *
   ***************************************************************************/
  case MOD_LOAD:
       printf(
               "Intel Pro Diagnostic Driver loading %d.%d.%d.%d\n" ,
               NAL_OS_SPEC_DRIVER_MAJOR_VERSION  ,
               NAL_OS_SPEC_DRIVER_MINOR_VERSION ,
               NAL_OS_SPEC_DRIVER_BUILD_VERSION ,
               NAL_OS_SPEC_DRIVER_FIX_VERSION );

       /* Clearing global arrays */
       memset(Global_PciAllocationSlotsTable, 0 , sizeof(Global_PciAllocationSlotsTable));
       memset(Global_DmaAllocationSlotsTable, 0 , sizeof(Global_DmaAllocationSlotsTable));
       memset(&Global_AtomicTestSetSpinLock , 0 , sizeof(Global_AtomicTestSetSpinLock));
       memset(Global_AdapterInUse           , 0 , sizeof(Global_AdapterInUse));

       mtx_init(&Global_AtomicTestSetSpinLock , "AtomicTestSetSpinLock" , NULL , MTX_DEF);

       /* The function below MUST be always AT THE END of module loading. */
       Global_NalDevicePointer = make_dev(
                                          &Global_DeviceEntryPoints ,   /* Driver interfaces */
                                          0 ,                           /* Index of module */
                                          UID_ROOT ,                    /* User-ID (create device as root user) */
                                          GID_WHEEL,                    /* Group-ID */
                                          0600 ,                        /* Permissions for device (only for root) */
                                          "nal" );                      /* Name of device */

       ErrorCode = 0;
       break;
   /***************************************************************************
   *
   * This case is executed when the NAL driver should be unloaded from the system.
   * It destroys all variables, free all allocated memory
   *
   ***************************************************************************/
  case MOD_UNLOAD:
       do
       {
           if(Global_NalDevicePointer == NULL)
           {
               /* Device not configured */
               ErrorCode = ENXIO;
               break;
           }

           if(Global_DriverReferenceCount > 0)
           {
               printf("Intel Pro Diagnostic Driver version %d.%d.%d.%d exiting with %d %s\n" ,
                       NAL_OS_SPEC_DRIVER_MAJOR_VERSION  ,
                       NAL_OS_SPEC_DRIVER_MINOR_VERSION ,
                       NAL_OS_SPEC_DRIVER_BUILD_VERSION ,
                       NAL_OS_SPEC_DRIVER_FIX_VERSION ,
                       Global_DriverReferenceCount ,
                       (Global_DriverReferenceCount==1) ? "reference" : "references");
           }

           /* The function below MUST be called before any unloading operations */
           destroy_dev(Global_NalDevicePointer);

           mtx_destroy(&Global_AtomicTestSetSpinLock);

           Global_NalDevicePointer  = NULL;
           ErrorCode                = 0;

       } while(0);
       break;
  }

  return ErrorCode;
}

/***************************************************************************
**
** Name:            NalOpenDriver()
**
** Description:     The function prepares the driver to perform ioctl
**                  commands.
**
** Arguments:       Device      - Pointer to device structure
**                  OpenFlags   - Flags of open mode
**                  DeviceType  - Type of the device
**                  Thread      - Pointer to a thread
**
** Returns:         Code of error from errno.h
**                      0           - when success
**
****************************************************************************/
int
NalOpenDriver(
              IN    struct cdev *   Device ,
              IN    int             OpenFlags ,
              IN    int             DeviceType ,
              IN    struct thread * Thread )
{
    NalDebugPrint("iqv: NalOpenDriver: Opening a connection with QV driver in version %s\n" , NAL_OS_SPEC_DRIVER_VERSION);
    return 0;
}

/***************************************************************************
**
** Name:            NalReleaseDriver()
**
** Description:     The function releases driver. It is called, when it
**                  should be closed.
**
** Arguments:       Device      - Pointer to device structure
**                  ReleaseFlags- Flags of close mode
**                  DeviceType  - Type of the device
**                  Thread      - Pointer to a thread
**
** Returns:         Code of error from errno.h
**                      0           - when success
**
****************************************************************************/
int
NalReleaseDriver(
                 IN     struct cdev *   Device ,
                 IN     int             ReleaseFlags  ,
                 IN     int             DeviceType ,
                 IN     struct thread * Thread )
{
    NalDebugPrint("iqv: NalReleaseDriver: Closing a connection with QV driver with %d references\n" , Global_DriverReferenceCount);
    return 0;
}

/***************************************************************************
**
** Name:            NalHandleIoctl()
**
** Description:     The function for handling IOCTL commands for NAL device.
**
** Arguments:       Device      - Pointer to device structure
**                  Command     - IOCTL command to execute
**                  Data        - Pointer to the data
**                  Flags       - Flags the command
**                  Thread      - Pointer to a thread
**
** Returns:         Code of error from errno.h
**                      0           - when success
**                      EDOOFUS     - when there is error in function implementation
**                      EOPNOTSUPP  - when the command is not recognized
**
****************************************************************************/
int
NalHandleIoctl(
               IN      struct cdev *   Device ,
               IN      u_long          Command ,
               IN OUT  caddr_t         Data ,
               IN      int             Flags ,
               IN      struct thread * Thread )
{
    int                   ErrorCode         = EDOOFUS;
    NAL_IOCTL_INPUT_DATA  InputData         = {};
    NAL_IOCTL_INPUT_DATA* NalIoctlInputData = (NAL_IOCTL_INPUT_DATA*)Data;

    /*************************************************************************
     *
     * The Data is not copied because the ioctl command has assigned size of
     * data, that is copied before call of this function
     *
     ************************************************************************/
    memset(&InputData , 0 , sizeof(InputData));

    switch(Command)
    {
    /*************************************************************************
     *
     * This case must be first. It occurs when the operation is not supported.
     *
     ************************************************************************/
    default:
        ErrorCode   = EOPNOTSUPP;
        break;
    /*************************************************************************
     *
     * Received OSI-type command
     *
     ************************************************************************/
    case IOCTL_NAL_OSI:
        NalResolveOsiIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;
    /*************************************************************************
     *
     * Received HW-BUS-type command
     *
     ************************************************************************/
    case IOCTL_NAL_HW_BUS:
        NalResolveHwBusIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;
    /*************************************************************************
     *
     * Received NDI-type command
     *
     ************************************************************************/
    case IOCTL_NAL_NDI:
        NalResolveNdiIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;

    /*************************************************************************
     *
     * Received FreeBSD specific command
     *
     ************************************************************************/
    case IOCTL_NAL_OS_SPECIFIC:
        NalResolveOsSpecificIoctl(NalIoctlInputData);
        ErrorCode   = 0;
        break;
    }

    return ErrorCode;
}

/***************************************************************************
**
** Name:            _NalDriverGetReferenceCount()
**
** Description:     This returns the reference count of the number of user mode components
**                  have referenced the driver.
**
** Arguments:       None
**
** Returns:         Number of references
**
****************************************************************************/
UINT32
_NalDriverGetReferenceCount(
    VOID
    )
{
    return Global_DriverReferenceCount;
}

/***************************************************************************
**
** Name:            _NalDriverIncrementReferenceCount()
**
** Description:     This increments the reference count of the number of user mode components
**                  have referenced the driver.
**
** Arguments:       nothing
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalDriverIncrementReferenceCount(
    VOID
    )
{
    /* Note, do not user the atomic increment here because it's intended for ring3 and will perform
     * a copy_from_user and copy_to_user on the address */
    mtx_lock(&Global_AtomicTestSetSpinLock);
    Global_DriverReferenceCount++;
    mtx_unlock(&Global_AtomicTestSetSpinLock);
}

/***************************************************************************
**
** Name:            _NalDriverDecrementReferenceCount()
**
** Description:     This decrements the reference count of the number of user mode components
**                  have referenced the driver.
**
** Arguments:       nothing
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalDriverDecrementReferenceCount(
    VOID
    )
{
    /* Note, do not user the atomic increment here because it's intended for ring3 and will perform
     * a copy_from_user and copy_to_user on the address */
    mtx_lock(&Global_AtomicTestSetSpinLock);
    if(Global_DriverReferenceCount > 0)
    {
        Global_DriverReferenceCount--;
    }
    mtx_unlock(&Global_AtomicTestSetSpinLock);
}

/***************************************************************************
**
** Name:            _NalDriverGetVersion()
**
** Description:     This returns the version number of the driver.
**
** Arguments:       None
**
** Returns:         version
**
****************************************************************************/
VOID
_NalDriverGetVersion(
    OUT CHAR* Version
    )
{
    sprintf(Version, "%d.%d.%d.%d",
                                   NAL_OS_SPEC_DRIVER_MAJOR_VERSION ,
                                   NAL_OS_SPEC_DRIVER_MINOR_VERSION ,
                                   NAL_OS_SPEC_DRIVER_BUILD_VERSION ,
                                   NAL_OS_SPEC_DRIVER_FIX_VERSION   );
}
